<?php
session_start();
header("Content-Type: application/json");

include "../../config/db.php";

/* ---------- AUTH CHECK ---------- */
if (!isset($_SESSION['uid'])) {
    http_response_code(401);
    echo json_encode([
        "status" => "error",
        "message" => "Session expired"
    ]);
    exit;
}

/* ---------- READ JSON ---------- */
$raw = file_get_contents("php://input");
$data = json_decode($raw, true);

if (!$data) {
    http_response_code(400);
    echo json_encode([
        "status" => "error",
        "message" => "Invalid JSON payload"
    ]);
    exit;
}

/* ---------- VALIDATE ---------- */
if (
    empty($data['emp_id']) ||
    empty($data['face_image'])
) {
    http_response_code(422);
    echo json_encode([
        "status" => "error",
        "message" => "Missing required fields"
    ]);
    exit;
}

$emp_id   = (int)$data['emp_id'];
$emp_name = trim($data['emp_name'] ?? '');
$createDT = date('Y-m-d H:i:s');

/* ---------- IMAGE SAVE ---------- */
$imgData = $data['face_image'];

// Validate base64 image
if (!preg_match('#^data:image/(jpeg|jpg|png);base64,#', $imgData)) {
    http_response_code(415);
    echo json_encode([
        "status" => "error",
        "message" => "Invalid image format"
    ]);
    exit;
}

$imgData = preg_replace('#^data:image/\w+;base64,#', '', $imgData);
$img = base64_decode($imgData);

if ($img === false) {
    http_response_code(400);
    echo json_encode([
        "status" => "error",
        "message" => "Image decode failed"
    ]);
    exit;
}

$dir = "../../uploads/faces";
if (!is_dir($dir)) {
    mkdir($dir, 0775, true);
}

if (!is_writable($dir)) {
    http_response_code(500);
    echo json_encode([
        "status" => "error",
        "message" => "Upload directory not writable"
    ]);
    exit;
}

$imgPath = $dir . "/emp_" . $emp_id . ".jpg";

if (!file_put_contents($imgPath, $img)) {
    http_response_code(500);
    echo json_encode([
        "status" => "error",
        "message" => "Failed to save image"
    ]);
    exit;
}

/* ---------- DB SAVE ---------- */
$stmt = $con->prepare("
    INSERT INTO face_recog (emp_id, emp_name, face_image, createDT)
    VALUES (?, ?, ?, ?)
    ON DUPLICATE KEY UPDATE
        emp_name = VALUES(emp_name),
        face_image = VALUES(face_image),
        createDT = VALUES(createDT)
");

$stmt->bind_param("isss", $emp_id, $emp_name, $imgPath, $createDT);

if (!$stmt->execute()) {
    http_response_code(500);
    echo json_encode([
        "status" => "error",
        "message" => "Database error"
    ]);
    exit;
}

/* ---------- RESPONSE ---------- */
echo json_encode([
    "status" => "ok"
]);