<?php
// salary_page.php  (replace your current page file with this content)
include('./include/header.php');
date_default_timezone_set('Asia/Kolkata');


// ---------------------------
// HELPERS
// ---------------------------
if (!function_exists('fetch_leave_settings')) {
    function fetch_leave_settings($con) {
        $settings = [];
        $res = $con->query("SELECT setting_key, setting_value FROM leave_settings");
        while ($r = $res->fetch_assoc()) {
            $settings[$r['setting_key']] = $r['setting_value'];
        }
        return $settings;
    }
}

if (!function_exists('getMonthName')) {
    function getMonthName($num) {
        $months = [1=>"January",2=>"February",3=>"March",4=>"April",5=>"May",6=>"June",
            7=>"July",8=>"August",9=>"September",10=>"October",11=>"November",12=>"December"];
        return $months[intval($num)] ?? "Unknown";
    }
}

/**
 * Compute salary total for one salary period using:
 *  - users.emp_join_date
 *  - leave_settings: weekly_off, casual_leave, sick_leave, leave_timeFrame
 *  - basic, incentive, other_incentive_amount, deductions_amount, manual_deduction
 *
 * Attendance integration is left as TODO placeholders (see commented sample).
 */
if (!function_exists('compute_salary_total')) {
    function compute_salary_total($con, $employee_id, $salary_month, $salary_year, $basic, $incentive = 0, $other_incentive_amount = 0, $deductions_amount = 0, $manual_deduction = 0) {
        // sanitize / cast
        $basic = floatval($basic);
        $incentive = floatval($incentive);
        $other_incentive_amount = floatval($other_incentive_amount);
        $deductions_amount = floatval($deductions_amount);
        $manual_deduction = floatval($manual_deduction);

        // 1) fetch join date
        $employee_id_esc = $con->real_escape_string($employee_id);
        $emp = $con->query("SELECT emp_join_date FROM users WHERE uid = '{$employee_id_esc}'")->fetch_assoc();
        $join_date = isset($emp['emp_join_date']) && $emp['emp_join_date'] ? $emp['emp_join_date'] : null;

        // 2) compute period start & end (1-month window from salary_month/year)
        // Default period start = first day of salary_month/year
        $period_start = DateTime::createFromFormat('!Y-n-j', "$salary_year-$salary_month-1");
        if ($period_start === false) {
            $period_start = new DateTime("$salary_year-$salary_month-01");
        }

        // If employee joined after period start, period_start = join_date
        if ($join_date) {
            $jd = DateTime::createFromFormat('!Y-m-d', $join_date);
            if ($jd && $jd > $period_start) {
                $period_start = clone $jd;
            }
        }

        // period_end = period_start + 1 month - 1 day
        $period_end = (clone $period_start)->modify('+1 month')->modify('-1 day');

        // total days in period (inclusive)
        $total_days = (int)$period_end->diff($period_start)->format('%a') + 1;

        // 3) fetch leave settings
        $ls = fetch_leave_settings($con);
        $weekly_off = isset($ls['weekly_off']) ? $ls['weekly_off'] : 'Sunday';
        $casual_leave = isset($ls['casual_leave']) ? intval($ls['casual_leave']) : 0;
        $sick_leave = isset($ls['sick_leave']) ? intval($ls['sick_leave']) : 0;
        $leave_timeFrame = isset($ls['leave_timeFrame']) ? $ls['leave_timeFrame'] : 'Monthly';

        // 4) count weekly offs within period
        $dow_map = [
            'Sunday'=>0,'Monday'=>1,'Tuesday'=>2,'Wednesday'=>3,'Thursday'=>4,'Friday'=>5,'Saturday'=>6
        ];
        $weekly_off_index = isset($dow_map[$weekly_off]) ? $dow_map[$weekly_off] : 0;

        $iter = clone $period_start;
        $weekly_offs = 0;
        for ($i = 0; $i < $total_days; $i++) {
            if ((int)$iter->format('w') === $weekly_off_index) $weekly_offs++;
            $iter->modify('+1 day');
        }

        // 5) working days in period (excluding weekly offs)
        $working_days = max(1, $total_days - $weekly_offs);

        // 6) Attendance: placeholder for future integration
        // TODO: Replace the following placeholders by querying your attendance table
        // and computing:
        //   $leaves_taken  => total full days of leave in period (absent / leave)
        //   $half_days_count => count of half-days (count them as 0.5)
        // Right now we assume zero leaves and zero half-days:
        $leaves_taken = 0;        // <-- plug real value here from attendance
        $half_days_count = 0;     // <-- plug real value (0.5 each)

        /* Sample code to integrate attendance if your table is named `employee_attendance`:
        $ps = $con->prepare("SELECT status, COUNT(*) as cnt FROM employee_attendance WHERE emp_id = ? AND att_date BETWEEN ? AND ? GROUP BY status");
        $start = $period_start->format('Y-m-d');
        $end = $period_end->format('Y-m-d');
        $ps->bind_param("sss", $employee_id, $start, $end);
        $ps->execute();
        $res = $ps->get_result();
        while ($r = $res->fetch_assoc()) {
            // assuming status values: 'Present','Absent','Leave','Half'
            if ($r['status'] == 'Leave' || $r['status'] == 'Absent') $leaves_taken += intval($r['cnt']);
            if ($r['status'] == 'Half') $half_days_count += intval($r['cnt']);
        }
        $ps->close();
        */

        // Convert half_days_count to fractional days
        $half_days_equivalent = floatval($half_days_count) * 0.5;

        // 7) allowed leaves based on timeframe (we support Monthly directly)
        if (strtolower($leave_timeFrame) === 'monthly') {
            $allowed_leaves = intval($casual_leave) + intval($sick_leave);
        } else {
            // If 'Yearly' or others - for now divide yearly quota by 12 (simple approximation)
            $allowed_leaves = floor((intval($casual_leave) + intval($sick_leave)) / 12);
        }

        // total leaves that consume pay (full-days + half-days counted)
        $total_leave_days = $leaves_taken + $half_days_equivalent;

        // leaves beyond allowed cause deduction
        $excess_leave_days = max(0, $total_leave_days - $allowed_leaves);

        // 8) prorate basic based on employment days in period
        $employment_days = $total_days; // default
        if ($join_date) {
            $join_dt = DateTime::createFromFormat('!Y-m-d', $join_date);
            if ($join_dt && $join_dt > (new DateTime("$salary_year-$salary_month-01"))) {
                // employment days = days from join_date to period_end inclusive
                $employment_days = (int)$period_end->diff($join_dt)->format('%a') + 1;
            }
        }

        // prorated basic based on employment days (not yet subtracting leave deductions)
        $basic_prorated = ($total_days > 0) ? ($basic * ($employment_days / $total_days)) : $basic;

        // 9) compute per-day rate (use full basic / total_days)
        $per_day_rate = ($total_days > 0) ? ($basic / $total_days) : 0;

        // 10) compute deductions for excess leaves
        $leave_deduction_amount = $excess_leave_days * $per_day_rate;

        // 11) final basic after leave deduction
        $final_basic_after_leave = max(0, $basic_prorated - $leave_deduction_amount);

        // 12) final total = basic_after_leave + incentive + other_incentive - existing deductions - manual deduction
        $final_total = $final_basic_after_leave + $incentive + $other_incentive_amount - $deductions_amount - $manual_deduction;

        // round to 2 decimals
        $final_total = round($final_total, 2);

        // return array with breakdown
        return [
            'final_total' => $final_total,
            'basic_prorated' => round($basic_prorated,2),
            'leave_deduction_amount' => round($leave_deduction_amount,2),
            'final_basic_after_leave' => round($final_basic_after_leave,2),
            'per_day_rate' => round($per_day_rate,2),
            'employment_days' => $employment_days,
            'total_days' => $total_days,
            'working_days' => $working_days,
            'period_start' => $period_start->format('Y-m-d'),
            'period_end' => $period_end->format('Y-m-d'),
            'allowed_leaves' => $allowed_leaves,
            'leaves_taken' => $leaves_taken,
            'half_days_count' => $half_days_count,
            'other_incentive_amount' => $other_incentive_amount,
            'deductions_amount' => $deductions_amount,
            'manual_deduction' => $manual_deduction
        ];
    }
}


// ---------------------------
// PAGE UI + LOGIC
// ---------------------------
?>
<div class="d-flex flex-column flex-lg-row">
    <?php include('./include/sidebar.php'); ?>
    <div class="flex-grow-1">
        <?php include('./include/topbar.php'); ?>

        <div class="container-fluid p-4" id="body-content">
            <?php
            // If employee_id provided, show employee salary page
            if (isset($_GET["employee_id"])) {
                $employee_id = intval($_GET["employee_id"]);
                $url = "employee-salary.php?employee_id=" . $employee_id;
                $find_sql = $con->query("SELECT * FROM users WHERE uid = '{$employee_id}'");
                $find_row = $find_sql->fetch_assoc();

                $months = [
                    1=>"January",2=>"February",3=>"March",4=>"April",5=>"May",6=>"June",
                    7=>"July",8=>"August",9=>"September",10=>"October",11=>"November",12=>"December"
                ];

                // -- HANDLERS: Add Salary --
                if(isset($_POST['add_salary'])) {
                    $currentTime = date('M j, Y g:i a');

                    $salary_month = mysqli_real_escape_string($con,strip_tags($_POST['salary_month']));
                    $salary_year  = mysqli_real_escape_string($con,strip_tags($_POST['salary_year']));
                    $salary_basic = floatval(mysqli_real_escape_string($con,strip_tags($_POST['slary_basic'])));
                    $salary_insentive = floatval(mysqli_real_escape_string($con,strip_tags($_POST['salary_insentive'])));
                    $other_insentive_title = mysqli_real_escape_string($con,strip_tags($_POST['other_insentive_title']));
                    $other_insentive_amount = floatval(mysqli_real_escape_string($con,strip_tags($_POST['other_insentive_amount'])));
                    $deductions_title = mysqli_real_escape_string($con,strip_tags($_POST['deductions_title']));
                    $deductions_amount = floatval(mysqli_real_escape_string($con,strip_tags($_POST['deductions_amount'])));
                    $salary_mode = mysqli_real_escape_string($con,strip_tags($_POST['salary_mode']));
                    $salary_transfar_id = mysqli_real_escape_string($con,strip_tags($_POST['salary_transfar_id']));
                    $salary_des = $_POST['content'] ?? '';
                    $manual_deduction = isset($_POST['manual_deduction']) ? floatval($_POST['manual_deduction']) : 0;

                    // Compute final total using helper
                    $salaryCalc = compute_salary_total($con, $employee_id, $salary_month, $salary_year, $salary_basic, $salary_insentive, $other_insentive_amount, $deductions_amount, $manual_deduction);
                    $salary_total = $salaryCalc['final_total'];

                    // Build INSERT - manual_deduction is NOT stored separately currently
                    $ins_sql = "INSERT INTO `employee_salary` 
                        (salary_for, salary_month, salary_year, salary_basic, salary_insentive, salary_total, salary_des, salary_mode, salary_transfar_id, salary_up_date, deductions_amount, deductions_title, other_insentive_amount, other_insentive_title)
                        VALUES ('{$employee_id}', '{$salary_month}', '{$salary_year}', '{$salary_basic}', '{$salary_insentive}', '{$salary_total}', '". $con->real_escape_string($salary_des) ."', '{$salary_mode}', '{$salary_transfar_id}', '{$currentTime}', '{$deductions_amount}', '". $con->real_escape_string($deductions_title) ."', '{$other_insentive_amount}', '". $con->real_escape_string($other_insentive_title) ."')";

                    if($con->query($ins_sql)){
                        echo "<script>window.location = '". $url ."';</script>";
                        exit;
                    } else {
                        echo "<script>alert('OOPS !!!!'); setTimeout(()=>{window.location='".$url."'},1200);</script>";
                        exit;
                    }
                }

                // -- HANDLERS: Edit Salary --
                if(isset($_POST['edit_salary'])) {
                    $currentTime = date('M j, Y g:i a');

                    $salary_month = mysqli_real_escape_string($con,strip_tags($_POST['salary_month']));
                    $salary_year  = mysqli_real_escape_string($con,strip_tags($_POST['salary_year']));
                    $salary_basic = floatval(mysqli_real_escape_string($con,strip_tags($_POST['slary_basic'])));
                    $salary_insentive = floatval(mysqli_real_escape_string($con,strip_tags($_POST['salary_insentive'])));
                    $other_insentive_title = mysqli_real_escape_string($con,strip_tags($_POST['other_insentive_title']));
                    $other_insentive_amount = floatval(mysqli_real_escape_string($con,strip_tags($_POST['other_insentive_amount'])));
                    $deductions_title = mysqli_real_escape_string($con,strip_tags($_POST['deductions_title']));
                    $deductions_amount = floatval(mysqli_real_escape_string($con,strip_tags($_POST['deductions_amount'])));
                    $salary_mode = mysqli_real_escape_string($con,strip_tags($_POST['salary_mode']));
                    $salary_transfar_id = mysqli_real_escape_string($con,strip_tags($_POST['salary_transfar_id']));
                    $salary_des = $_POST['content'] ?? '';
                    $manual_deduction = isset($_POST['manual_deduction']) ? floatval($_POST['manual_deduction']) : 0;

                    // Compute final total
                    $salaryCalc = compute_salary_total($con, $employee_id, $salary_month, $salary_year, $salary_basic, $salary_insentive, $other_insentive_amount, $deductions_amount, $manual_deduction);
                    $salary_total = $salaryCalc['final_total'];

                    if(isset($_GET['edit_id'])) {
                        $edit_id = intval($_GET['edit_id']);
                        $update_sql = "UPDATE `employee_salary` SET
                            `salary_month` = '{$salary_month}',
                            `salary_year`  = '{$salary_year}',
                            `other_insentive_title` = '". $con->real_escape_string($other_insentive_title) ."',
                            `other_insentive_amount` = '{$other_insentive_amount}',
                            `deductions_title` = '". $con->real_escape_string($deductions_title) ."',
                            `deductions_amount` = '{$deductions_amount}',
                            `salary_basic` = '{$salary_basic}',
                            `salary_insentive` = '{$salary_insentive}',
                            `salary_total` = '{$salary_total}',
                            `salary_des` = '". $con->real_escape_string($salary_des) ."',
                            `salary_mode` = '{$salary_mode}',
                            `salary_transfar_id` = '{$salary_transfar_id}',
                            `salary_edit_date` = '{$currentTime}'
                            WHERE `emp_salary_id` = {$edit_id}";

                        if($con->query($update_sql)){
                            echo "<script>window.location = '". $url ."';</script>";
                            exit;
                        } else {
                            echo "<script>alert('Update failed! Please try again.'); window.location = '". $url ."';</script>";
                            exit;
                        }
                    }
                }

                // -- HANDLERS: Delete --
                if ( isset($_GET["del_id"]) ) {
                    $delete_id = intval($_GET["del_id"]);
                    $delete_sql = "DELETE FROM employee_salary WHERE emp_salary_id = '{$delete_id}'";
                    if ($con->query($delete_sql)) {
                        echo "<script>window.location = '{$url}';</script>";
                        exit;
                    } else {
                        echo "<script>alert('OOPS !!!!'); setTimeout(()=>{window.location='{$url}'},1200);</script>";
                        exit;
                    }
                }

                // Display edit form if edit_id set
                if (isset($_GET["edit_id"])) {
                    $edit_id = intval($_GET["edit_id"]);
                    $sql = $con->query("SELECT * FROM employee_salary WHERE emp_salary_id = '{$edit_id}'");
                    if ($row = $sql->fetch_assoc()) {
                        $monthNumber = $row["salary_month"];
                        $monthName = getMonthName($monthNumber);
                        ?>
            <!-- EDIT SALARY FORM -->
            <div class="card shadow-sm p-3 mb-4">
                <h3 class="text-center">Edit Salary: <?php echo htmlspecialchars($find_row["uname"]); ?></h3>
                <form method="post" enctype="multipart/form-data">
                    <div class="row">
                        <div class="col-md-6 mb-1">
                            <label>Select Month:</label>
                            <select name="salary_month" class="form-control">
                                <option value="<?php echo htmlspecialchars($row["salary_month"]); ?>">
                                    <?php echo htmlspecialchars($monthName); ?></option>
                                <?php foreach ($months as $num=>$name) echo "<option value='".intval($num)."'>".htmlspecialchars($name)."</option>"; ?>
                            </select>
                        </div>
                        <div class="col-md-6 mb-1">
                            <label>Select Year:</label>
                            <select name="salary_year" class="form-control" id="year2">
                                <option value="<?php echo htmlspecialchars($row["salary_year"]); ?>">
                                    <?php echo htmlspecialchars($row["salary_year"]); ?></option>
                            </select>
                        </div>
                        <div class="col-md-6 mb-1">
                            <label>Basic Salary</label>
                            <input type="text" class="form-control form-control-sm" name="slary_basic"
                                value="<?php echo htmlspecialchars($row["salary_basic"]); ?>" required>
                        </div>
                        <div class="col-md-6 mb-1">
                            <label>Incentive Salary</label>
                            <input type="text" class="form-control form-control-sm" name="salary_insentive"
                                value="<?php echo htmlspecialchars($row["salary_insentive"]); ?>" required>
                        </div>
                        <div class="col-md-6 mb-1">
                            <label>Other Incentive Title</label>
                            <input type="text" class="form-control form-control-sm" name="other_insentive_title"
                                value="<?php echo htmlspecialchars($row["other_insentive_title"]); ?>" required>
                        </div>
                        <div class="col-md-6 mb-1">
                            <label>Other Incentive Amount</label>
                            <input type="text" class="form-control form-control-sm" name="other_insentive_amount"
                                value="<?php echo htmlspecialchars($row["other_insentive_amount"]); ?>" required>
                        </div>
                        <div class="col-md-6 mb-1">
                            <label>Deduction Title</label>
                            <input type="text" class="form-control form-control-sm" name="deductions_title"
                                value="<?php echo htmlspecialchars($row["deductions_title"]); ?>" required>
                        </div>
                        <div class="col-md-6 mb-1">
                            <label>Deduction Amount</label>
                            <input type="text" class="form-control form-control-sm" name="deductions_amount"
                                value="<?php echo htmlspecialchars($row["deductions_amount"]); ?>" required>
                        </div>
                        <div class="col-md-6 mb-1">
                            <label>Manual Deduction (optional)</label>
                            <input type="number" step="0.01" class="form-control form-control-sm"
                                name="manual_deduction" value="0">
                            <small class="text-muted">Ad-hoc deduction applied at calculation time (not stored).</small>
                        </div>
                        <div class="col-md-6 mb-1">
                            <label>Salary Mode</label>
                            <select name="salary_mode" class="form-select form-control">
                                <option value="<?php echo htmlspecialchars($row["salary_mode"]); ?>" selected>
                                    <?php echo htmlspecialchars($row["salary_mode"]); ?></option>
                                <option value="Cash">Cash</option>
                                <option value="G-pay">G-pay</option>
                                <option value="PhonePay">PhonePay</option>
                                <option value="Paytm">Paytm</option>
                                <option value="Bank">Bank</option>
                            </select>
                        </div>
                        <div class="col-md-6 mb-1">
                            <label>Salary Transfer ID</label>
                            <input type="text" class="form-control form-control-sm" name="salary_transfar_id"
                                value="<?php echo htmlspecialchars($row["salary_transfar_id"]); ?>">
                        </div>
                        <div class="col-sm-12 mb-1">
                            <label>Enter Salary Details:</label>
                            <textarea class="form-control ckeditor" name="content"
                                style="min-height:100px;"><?php echo htmlspecialchars($row["salary_des"]); ?></textarea>
                        </div>
                        <div class="mt-3">
                            <button type="submit" class="btn btn-primary my-2 btn-sm" name="edit_salary">
                                <i class="fa fa-edit"></i> Update Salary
                            </button>
                            <a href="<?php echo $url; ?>" class="btn btn-warning m-2 btn-sm">
                                <i class="fa fa-ban"></i> Cancel
                            </a>
                        </div>
                    </div>
                </form>
            </div>
            <?php
                    } // end fetch row
                } else {
                    // ADD Salary UI and Table
                    ?>
            <!-- ADD SALARY -->
            <div class="card shadow-sm p-3 mb-4">
                <h5>Salary for <?php echo htmlspecialchars($find_row["uname"]); ?></h5>
                <form method="post" enctype="multipart/form-data">
                    <h6 class="badge bg-secondary mb-2" data-bs-toggle="collapse" data-bs-target="#salaryForm"
                        style="cursor:pointer;">
                        <i class="fa-solid fa-hand-pointer"></i> Add Salary Details
                    </h6>

                    <div class="collapse" id="salaryForm">
                        <div class="row g-3">
                            <div class="col-md-4">
                                <label>Select Month</label>
                                <select name="salary_month" class="form-select" required>
                                    <?php foreach ($months as $num=>$name) echo "<option value='".intval($num)."'>".htmlspecialchars($name)."</option>"; ?>
                                </select>
                            </div>
                            <div class="col-md-4">
                                <label>Select Year</label>
                                <select name="salary_year" id="year" class="form-select" required></select>
                            </div>
                            <div class="col-md-4">
                                <label>Basic Salary</label>
                                <input type="text" class="form-control" name="slary_basic" required>
                            </div>
                            <div class="col-md-4">
                                <label>Incentive Salary</label>
                                <input type="text" class="form-control" name="salary_insentive" required>
                            </div>
                            <div class="col-md-6">
                                <label>Other Incentive Title</label>
                                <input type="text" class="form-control" name="other_insentive_title" required>
                            </div>
                            <div class="col-md-6">
                                <label>Other Incentive Amount</label>
                                <input type="text" class="form-control" name="other_insentive_amount" required>
                            </div>
                            <div class="col-md-6">
                                <label>Deduction Title</label>
                                <input type="text" class="form-control" name="deductions_title" required>
                            </div>
                            <div class="col-md-6">
                                <label>Deduction Amount</label>
                                <input type="text" class="form-control" name="deductions_amount" required>
                            </div>

                            <div class="col-md-6">
                                <label>Manual Deduction (optional)</label>
                                <input type="number" step="0.01" class="form-control" name="manual_deduction" value="0">
                                <small class="text-muted">Ad-hoc deduction applied at calculation time (not
                                    stored).</small>
                            </div>

                            <div class="col-md-6">
                                <label>Salary Mode</label>
                                <select name="salary_mode" class="form-select" required>
                                    <option value="">Select Payment Mode</option>
                                    <option value="Cash">Cash</option>
                                    <option value="G-pay">G-Pay</option>
                                    <option value="PhonePay">PhonePe</option>
                                    <option value="Paytm">Paytm</option>
                                    <option value="Bank">Bank</option>
                                </select>
                            </div>
                            <div class="col-md-6">
                                <label>Salary Transfer ID</label>
                                <input type="text" class="form-control" name="salary_transfar_id">
                            </div>
                            <div class="col-12">
                                <label>Salary Details</label>
                                <textarea class="form-control ckeditor" name="content"
                                    style="min-height:100px;"></textarea>
                            </div>
                        </div>

                        <div class="mt-4">
                            <button type="submit" name="add_salary" class="btn btn-primary btn-sm">
                                <i class="fa fa-plus"></i> Add Salary
                            </button>
                            <button type="button" class="btn btn-warning btn-sm" id="cancelSalaryForm">
                                <i class="fa fa-ban"></i> Cancel
                            </button>
                        </div>
                    </div>
                </form>
                <div>
                    <a href="employees.php" class="btn btn-warning btn-sm my-2"><i class="fa fa-ban"></i> Cancel</a>
                </div>
            </div>

            <!-- SALARY TABLE -->
            <div class="card shadow-sm p-3 mb-4">
                <div class="table-responsive">
                    <table id="salaryTable" class="table table-striped table-bordered">
                        <thead>
                            <tr>
                                <th>#</th>
                                <th>Year/Month</th>
                                <th>Basic</th>
                                <th>Incentive</th>
                                <th>Other Incentive</th>
                                <th>Deductions</th>
                                <th>Total</th>
                                <th>Mode</th>
                                <th>Payment Date</th>
                                <th>Action</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php
                                    $y_sql = $con->query("SELECT DISTINCT salary_year FROM employee_salary WHERE salary_for = '{$employee_id}' ORDER BY emp_salary_id DESC");
                                    $a = 1;
                                    while ($y_row = $y_sql->fetch_assoc()) {
                                        $sql = $con->query("SELECT * FROM employee_salary WHERE salary_for = '{$employee_id}' AND salary_year = '{$y_row['salary_year']}' ORDER BY emp_salary_id DESC");
                                        while ($row = $sql->fetch_assoc()) {
                                            $monthName = getMonthName($row["salary_month"]);
                                            ?>
                            <tr>
                                <td><?php echo $a++; ?></td>
                                <td><?php echo htmlspecialchars($y_row["salary_year"]); ?> -
                                    <?php echo htmlspecialchars($monthName); ?></td>
                                <td><?php echo htmlspecialchars($row["salary_basic"]); ?></td>
                                <td><?php echo htmlspecialchars($row["salary_insentive"]); ?></td>
                                <td><?php echo htmlspecialchars($row["other_insentive_title"])." : ".htmlspecialchars($row["other_insentive_amount"]); ?>
                                </td>
                                <td><?php echo htmlspecialchars($row["deductions_title"])." : ".htmlspecialchars($row["deductions_amount"]); ?>
                                </td>
                                <td><?php echo htmlspecialchars($row["salary_total"]); ?></td>
                                <td><?php echo htmlspecialchars($row["salary_mode"]); ?></td>
                                <td><?php echo htmlspecialchars($row["salary_up_date"]); ?></td>
                                <td>
                                    <a href="<?php echo $url;?>&edit_id=<?php echo $row["emp_salary_id"];?>"
                                        class="btn btn-success btn-sm" title="Click to Edit">
                                        <i class="fa fa-pencil-square" aria-hidden="true" style="font-size:17px;"></i>
                                    </a>
                                    <a href="<?php echo $url;?>&del_id=<?php echo $row["emp_salary_id"];?>"
                                        class="delete btn btn-danger btn-sm" title="Click to Delete"
                                        onclick="return confirm('Delete this salary record?');">
                                        <i class="fa fa-trash" aria-hidden="true" style="font-size:19px;"></i>
                                    </a>
                                    <button class="btn btn-primary btn-sm print-btn"
                                        data-id="<?php echo $row["emp_salary_id"]; ?>">Print Slip</button>
                                </td>
                            </tr>
                            <?php
                                        }
                                    }
                                    ?>
                        </tbody>
                    </table>
                </div>
            </div>
            <?php
                } // end else (not edit)
            } // end if employee_id
            ?>
        </div>

        <?php include('./include/footer.php'); ?>
    </div>
</div>

<!--  SALARY PRINT MODAL  -->
<div class="modal fade" id="printModal" tabindex="-1" aria-labelledby="printModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg modal-dialog-centered modal-dialog-scrollable">
        <div class="modal-content" id="printArea">
            <div class="modal-header">
                <h5 class="modal-title" id="printModalLabel">Salary Slip</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body" id="printContent">
                <div class="text-center text-muted my-4">Loading...</div>
            </div>
            <div class="modal-footer d-print-none">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal"><i class="fa fa-times"></i>
                    Close</button>
                <button type="button" class="btn btn-primary" id="printBtn"><i class="fa fa-print"></i> Print</button>
            </div>
        </div>
    </div>
</div>

<!-- SCRIPT ORDER -->
<script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
<link rel="stylesheet" href="https://cdn.datatables.net/1.13.8/css/dataTables.bootstrap5.min.css">
<script src="https://cdn.datatables.net/1.13.8/js/jquery.dataTables.min.js"></script>
<script src="https://cdn.datatables.net/1.13.8/js/dataTables.bootstrap5.min.js"></script>
<script src="../assets/vendor/ckeditor.js"></script>

<script>
CKEDITOR.replace('content', {
    height: 300,
    filebrowserUploadUrl: "./ajax/ckEditorFileUpload.php"
});
document.addEventListener("DOMContentLoaded", function() {
    if ($.fn.DataTable && $('#salaryTable').length) {
        $('#salaryTable').DataTable({
            responsive: true,
            pageLength: 10
        });
    }

    const yearSelect = document.getElementById("year");
    if (yearSelect) {
        const currentYear = new Date().getFullYear();
        for (let y = currentYear; y >= currentYear - 10; y--) {
            const opt = document.createElement("option");
            opt.value = y;
            opt.textContent = y;
            yearSelect.appendChild(opt);
        }
    }

    const year2 = document.getElementById("year2");
    if (year2) {
        const currentYear = new Date().getFullYear();
        for (let y = currentYear; y >= currentYear - 10; y--) {
            const opt = document.createElement("option");
            opt.value = y;
            opt.textContent = y;
            year2.appendChild(opt);
        }
    }

    // Handle Print Slip Button
    $(document).on('click', '.print-btn', function() {
        const salaryId = $(this).data('id');
        $('#printContent').html('<div class="text-center text-muted my-4">Loading...</div>');
        const modal = new bootstrap.Modal(document.getElementById('printModal'));
        modal.show();

        $.get('./ajax/get-salary-slip.php', {
            id: salaryId
        }, function(data) {
            $('#printContent').html(data);
        }).fail(function() {
            $('#printContent').html(
                '<p class="text-danger text-center">Unable to load slip.</p>');
        });
    });

    // Print Button Action (print modal content only)
    $('#printBtn').on('click', function() {
        const printContents = document.getElementById('printArea').innerHTML;
        const originalContents = document.body.innerHTML;
        document.body.innerHTML = printContents;
        window.print();
        document.body.innerHTML = originalContents;
        location.reload();
    });

    // Cancel collapse
    const cancelBtn = document.getElementById('cancelSalaryForm');
    if (cancelBtn) {
        cancelBtn.addEventListener('click', function() {
            const collapseElement = document.getElementById('salaryForm');
            const bsCollapse = bootstrap.Collapse.getOrCreateInstance(collapseElement);
            bsCollapse.hide();
        });
    }
});
</script>

<style>
@media print {
    body * {
        visibility: hidden;
    }

    #printArea,
    #printArea * {
        visibility: visible;
    }

    #printArea {
        position: absolute;
        left: 0;
        top: 0;
        width: 100%;
    }
}
</style>

<?php include('./include/js.php'); ?>


<?php
// ---------------------------
// Additional notes for implementer
// ---------------------------

/*
1) Attendance integration:
   - If you have an attendance table (e.g. employee_attendance with columns emp_id, att_date, status),
     remove the placeholder inside compute_salary_total() and run queries to compute:
       - $leaves_taken  (count of Leave/Absent)
       - $half_days_count (count of Half)
     I provided a sample commented prepared-statement earlier in the helper.

2) Printing slip details:
   - In your ./ajax/get-salary-slip.php, fetch the employee_salary row, then call compute_salary_total()
     with that row's month/year/basic/incentive/deductions to display breakdown:
       $calc = compute_salary_total($con, $row['salary_for'], $row['salary_month'], $row['salary_year'], $row['salary_basic'], $row['salary_insentive'], $row['other_insentive_amount'], $row['deductions_amount'], 0);
     Then print $calc['period_start'], $calc['period_end'], $calc['basic_prorated'], $calc['leave_deduction_amount'], $calc['final_total'] etc.

3) Manual Deduction persistence:
   - Currently manual_deduction is applied to calculation but NOT stored in DB.
   - If you want to store it, ALTER TABLE employee_salary ADD COLUMN manual_deduction DECIMAL(12,2) DEFAULT 0;
     and include it in INSERT/UPDATE statements.

4) Weekly off logic:
   - The compute function excludes the configured weekly_off (from leave_settings) from working days.

5) Half-day:
   - compute_salary_total treats each half-day as 0.5 day. You must populate $half_days_count using attendance data.

6) Prorating span (join-date logic):
   - The period start is the first day of salary_month OR the join date if joined after that first day.
   - Period end is start + 1 month - 1 day, so joining Mar 10 -> period Mar10 - Apr9

7) Security:
   - I used escaping and int casting in places; adapt prepared statements for stronger security if desired.

If you want, I can now:
 - plug attendance queries (send your attendance table schema),
 - persist manual_deduction (I'll provide the ALTER TABLE and update SQL),
 - update get-salary-slip.php to show full breakdown.
*/
?>