<?php
include("config/functions.php");

$message = "";
$success = false;
$debug = true;
$debugLog = [];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $method = clean($_POST['method']);
    $identifier = trim(clean($_POST['identifier'] ?? ''));

    $debugLog[] = "POST method = {$method}";
    $debugLog[] = "Identifier = {$identifier}";

    if ($method === 'email') {
        if (!preg_match("/^[A-Z0-9._%+-]+@[A-Z0-9.-]+\.[A-Z]{2,}$/i", $identifier)) {
            $message = "Please enter a valid email address.";
            $debugLog[] = "Email validation failed for {$identifier}";
        } else {
            $debugLog[] = "Email validation passed";

            $stmt = $con->prepare("SELECT * FROM users WHERE umail = ?");
            if (!$stmt) $debugLog[] = "SQL prepare failed: " . $con->error;
            $stmt->bind_param("s", $identifier);
            $stmt->execute();
            $user = $stmt->get_result()->fetch_assoc();

            if ($user) {
                $debugLog[] = "User found: UID {$user['uid']}";

                $token = bin2hex(random_bytes(32));
                $expiry = date("Y-m-d H:i:s", strtotime("+30 minutes"));

                $stmt = $con->prepare("UPDATE users SET reset_token=?, reset_expiry=? WHERE uid=?");
                $stmt->bind_param("ssi", $token, $expiry, $user['uid']);
                $stmt->execute();

                $resetLink = BASE_URL . "HRM/reset-password.php?token=" . urlencode($token);
                $debugLog[] = "Reset link: {$resetLink}";

                $subject = "HRM Password Reset";
                $messageBody = "Dear " . htmlspecialchars($user['uname']) . ",\n\n";
                $messageBody .= "Click below to reset your password:\n{$resetLink}\n\n";
                $messageBody .= "If you didn’t request this, please ignore this email.";

                $headers = "From: no-reply@yourdomain.com\r\n";

                if (mail($user['umail'], $subject, $messageBody, $headers)) {
                    $message = "A password reset link has been sent to your registered email.";
                    $success = true;
                    $debugLog[] = "Mail sent successfully to {$user['umail']}";
                } else {
                    $message = "Failed to send reset email.";
                    $debugLog[] = "Mail failed";
                }
            } else {
                $message = "No account found with this email.";
                $debugLog[] = "No user found for {$identifier}";
            }
        }
    } elseif ($method === 'mobile') {
        if (!preg_match('/^[0-9]{10}$/', $identifier)) {
            $message = "Please enter a valid 10-digit mobile number.";
            $debugLog[] = "Invalid mobile format";
        } else {
            $stmt = $con->prepare("SELECT * FROM users WHERE uphone = ?");
            $stmt->bind_param("s", $identifier);
            $stmt->execute();
            $user = $stmt->get_result()->fetch_assoc();

            if ($user) {
                $message = "Password reset via mobile is currently not available. Please use your registered email.";
                $debugLog[] = "User found by mobile ({$identifier}), but SMS not available.";
            } else {
                $message = "No account found with this mobile number.";
                $debugLog[] = "No user found for mobile: {$identifier}";
            }
        }
    } else {
        $message = "Please select a valid reset method.";
        $debugLog[] = "Invalid method";
    }
}
?>
<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Forgot Password | HRM Dashboard</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css" rel="stylesheet">
    <style>
    body {
        background: linear-gradient(135deg, #007bff 0%, #00bcd4 100%);
        height: 100vh;
        display: flex;
        align-items: center;
        justify-content: center;
    }

    .login-card {
        width: 100%;
        max-width: 420px;
        background: #fff;
        border-radius: 15px;
        box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
        padding: 2rem;
    }

    .method-btns .btn {
        border-radius: 30px;
        font-weight: 500;
    }

    .method-btns .btn.active {
        color: #fff;
    }

    pre.debug {
        background: #212529;
        color: #0f0;
        padding: 10px;
        font-size: 12px;
        border-radius: 8px;
        overflow-x: auto;
    }
    </style>
</head>

<body>
    <div class="login-card">
        <div class="text-center mb-3">
            <h4 class="text-primary mb-1">Forgot Password</h4>
            <p class="text-muted small mb-0">Reset your HRM account password</p>
        </div>

        <?php if ($message): ?>
        <div class="alert alert-<?= $success ? 'success' : 'danger' ?> small text-center py-2">
            <?= htmlspecialchars($message) ?></div>
        <?php endif; ?>

        <form method="POST" id="resetForm" autocomplete="off">
            <div class="method-btns d-flex justify-content-center gap-2 mb-3">
                <button type="button" class="btn btn-outline-primary px-3" id="selectEmail"><i
                        class="fa fa-envelope me-1"></i>Email</button>
                <button type="button" class="btn btn-outline-success px-3" id="selectMobile"><i
                        class="fa fa-phone me-1"></i>Mobile</button>
            </div>

            <input type="hidden" name="method" id="method">
            <input type="hidden" name="identifier" id="identifierHidden">

            <div class="form-floating mb-3" id="emailField" style="display:none;">
                <input type="text" class="form-control" id="emailInput" placeholder="Enter Email">
                <label for="emailInput"><i class="fa fa-envelope me-2"></i>Email Address</label>
            </div>

            <div class="form-floating mb-3" id="mobileField" style="display:none;">
                <input type="text" class="form-control" id="mobileInput" placeholder="Enter Mobile Number"
                    maxlength="10">
                <label for="mobileInput"><i class="fa fa-phone me-2"></i>Mobile Number</label>
            </div>

            <button type="submit" class="btn btn-primary w-100 py-2" id="resetBtn" style="display:none;">Send Reset
                Link</button>
        </form>

        <div class="text-center mt-3">
            <a href="login.php" class="text-decoration-none text-primary small"><i
                    class="fa fa-arrow-left me-1"></i>Back to Login</a>
        </div>

        <?php if ($debug && count($debugLog) > 0): ?>
        <h6 class="mt-3 text-muted small">Debug Info:</h6>
        <pre class="debug"><?= htmlspecialchars(implode("\n", $debugLog)) ?></pre>
        <?php endif; ?>
    </div>

    <script>
    const selectEmail = document.getElementById('selectEmail');
    const selectMobile = document.getElementById('selectMobile');
    const emailField = document.getElementById('emailField');
    const mobileField = document.getElementById('mobileField');
    const methodInput = document.getElementById('method');
    const resetBtn = document.getElementById('resetBtn');
    const hiddenIdentifier = document.getElementById('identifierHidden');
    const emailInput = document.getElementById('emailInput');
    const mobileInput = document.getElementById('mobileInput');

    function hideAll() {
        emailField.style.display = 'none';
        mobileField.style.display = 'none';
        resetBtn.style.display = 'none';
        selectEmail.classList.remove('btn-primary', 'active');
        selectEmail.classList.add('btn-outline-primary');
        selectMobile.classList.remove('btn-success', 'active');
        selectMobile.classList.add('btn-outline-success');
    }

    selectEmail.addEventListener('click', () => {
        hideAll();
        methodInput.value = 'email';
        emailField.style.display = 'block';
        resetBtn.style.display = 'block';
        selectEmail.classList.remove('btn-outline-primary');
        selectEmail.classList.add('btn-primary', 'active');
    });

    selectMobile.addEventListener('click', () => {
        hideAll();
        methodInput.value = 'mobile';
        mobileField.style.display = 'block';
        resetBtn.style.display = 'block';
        selectMobile.classList.remove('btn-outline-success');
        selectMobile.classList.add('btn-success', 'active');
    });

    // Copy input value into hidden field on submit
    document.getElementById('resetForm').addEventListener('submit', function(e) {
        if (methodInput.value === 'email') hiddenIdentifier.value = emailInput.value.trim();
        else if (methodInput.value === 'mobile') hiddenIdentifier.value = mobileInput.value.trim();
    });
    </script>
</body>

</html>